#!/usr/bin/env bash
# menu.sh – egyszerű menü sorszámozott alprogramokhoz (01-*.sh … 09-*.sh)

set -u

SCRIPTS_DIR="${SCRIPTS_DIR:-.}"

# Sorszámozott (NN-*.sh) fájlok összegyűjtése rendezve
list_scripts() {
  # Csak a NN-kezdetű .sh fájlokat vesszük (01-*.sh … 99-*.sh)
  find "$SCRIPTS_DIR" -maxdepth 1 -type f -regex '.*/[0-9][0-9]-.*\.sh' \
    | sort
}

# Szép név a menühöz: "NN - Név"
pretty_name() {
  local f="$1"
  local base
  base="$(basename "$f")"
  # Leválasztjuk az elejét: "NN-" és a ".sh"-t
  local nn="${base:0:2}"
  local name="${base:3}"
  name="${name%.sh}"
  echo "$nn - $name"
}

pause() {
  echo
  read -r -p "Nyomj Enter-t a folytatáshoz..." _
}

run_script() {
  local f="$1"
  clear
  echo "=== Fut: $(pretty_name "$f") ==="
  echo

  # Nem követelünk +x jogot, közvetlenül a bash-sel futtatjuk
  bash "$f"
  local rc=$?

  echo
  echo "=== Kilépési kód: $rc ==="
  pause
}

main_menu() {
  while : ; do
    clear
    echo "============================"
    echo "     SZOLGÁLTATÁS MENÜ      "
    echo "============================"

    # Friss lista minden körben
    mapfile -t SCRIPTS < <(list_scripts)

    if (( ${#SCRIPTS[@]} == 0 )); then
      echo "Nincs futtatható alprogram (NN-*.sh) a mappában."
      echo "Másold ide a scripteket, majd indítsd újra."
      echo "0) Kilépés"
      echo "============================"
      read -r -p "Választás: " ch
      [[ "$ch" == "0" ]] && exit 0 || continue
    fi

    # Kiírás sorszámmal: 1..N
    local i=1
    for f in "${SCRIPTS[@]}"; do
      printf "%d) %s\n" "$i" "$(pretty_name "$f")"
      ((i++))
    done
    echo "0) Kilépés"
    echo "============================"
    read -r -p "Válassz (0-${#SCRIPTS[@]}): " ch

    # Kilépés
    if [[ "$ch" == "0" ]]; then
      echo "Kilépés..."
      exit 0
    fi

    # Szám-e és tartományon belül?
    if [[ "$ch" =~ ^[0-9]+$ ]] && (( ch>=1 && ch<=${#SCRIPTS[@]} )); then
      run_script "${SCRIPTS[ ch-1 ]}"
    else
      echo "Érvénytelen választás!"
      pause
    fi
  done
}

main_menu

